import numpy as np

class UnitConverter:
    """
    Unit conversion class for TDS data.
    Handles temperature, flux, concentration, and delta C conversions.
    """
    
    def __init__(self, atw=1.008, mass_density=7.8474, thickness=0.0063):
        """
        Initialize converter with material properties.
        
        Args:
            atw: Atomic weight of hydrogen (g/mol), default 1.008
            mass_density: Material mass density (g/cm³), default 7.8474
            thickness: Specimen thickness (m), default 0.0063
        """
        self.atw = atw
        self.mass_density = mass_density
        self.thickness = thickness
    
    def temperature_to_standard(self, temp, from_unit):
        """
        Convert temperature to standard units (K).
        
        Args:
            temp: Temperature array/value
            from_unit: Source unit ('K' or '°C')
        
        Returns:
            Temperature in Kelvin
        """
        temp = np.array(temp)
        
        if from_unit == 'K':
            return temp
        elif from_unit == '\u0394C':
            return temp + 273.15
        else:
            raise ValueError(f"Unknown temperature unit: {from_unit}")
    
    def temperature_from_standard(self, temp, to_unit):
        """
        Convert temperature from standard units (K) to desired units.
        
        Args:
            temp: Temperature array/value in Kelvin
            to_unit: Target unit ('K' or '°C')
        
        Returns:
            Temperature in target units
        """
        temp = np.array(temp)
        
        if to_unit == 'K':
            return temp
        elif to_unit == '\u0394C':
            return temp - 273.15
        else:
            raise ValueError(f"Unknown temperature unit: {to_unit}")
    
    def flux_to_standard(self, flux, from_unit):
        """
        Convert flux to standard units (mol/m²/s).
        
        Args:
            flux: Flux array/value
            from_unit: Source unit ('mol/m2s', 'mol/cm2s', 'wppm m/s')
        
        Returns:
            Flux in mol/m²/s
        """
        flux = np.array(flux)
        
        if from_unit == 'mol/m\u00b2s':
            return flux
        elif from_unit == 'mol/cm\u00b2s':
            return flux * 1e4  # Convert from cm² to m²
        elif from_unit == 'wppm m/s':
            coef = (1.01/self.atw) * (self.atw/self.mass_density)
            return flux / coef
        else:
            raise ValueError(f"Unknown flux unit: {from_unit}")
    
    def flux_from_standard(self, flux, to_unit):
        """
        Convert flux from standard units (mol/m²/s) to desired units.
        
        Args:
            flux: Flux array/value in mol/m²/s
            to_unit: Target unit ('mol/m2s', 'mol/cm2s', 'wppm m/s')
        
        Returns:
            Flux in target units
        """
        flux = np.array(flux)
        
        if to_unit == 'mol/m\u00b2s':
            return flux
        elif to_unit == 'mol/cm\u00b2s':
            return flux * 1e-4 
        elif to_unit == 'wppm m/s':
            coef = (1.01/self.atw) * (self.atw/self.mass_density)
            return flux * coef
        else:
            raise ValueError(f"Unknown flux unit: {to_unit}")
    
    def delta_c_to_standard(self, delta_c, from_unit):
        """
        Convert delta C to standard units (mol/m3s).
        
        Args:
            delta_c: Delta C array/value
            from_unit: Source unit ('mol/m3s', 'mol/cm3s', 'wppm/s')
        
        Returns:
            Delta C in mol/m3s
        """
        delta_c = np.array(delta_c)
        
        if from_unit == 'mol/m\u00b3s':
            return delta_c
        elif from_unit == 'mol/cm\u00b3s':
            return delta_c * 1e6  
        elif from_unit == 'wppm/s':
            coef = (1.01/self.atw) * (self.atw/self.mass_density)
            return delta_c / coef
        else:
            raise ValueError(f"Unknown delta C unit: {from_unit}")
    
    def delta_c_from_standard(self, delta_c, to_unit):
        """
        Convert delta C from standard units (mol/m3/s) to desired units.
        
        Args:
            delta_c: Delta C array/value in mol/m3s
            to_unit: Target unit ('mol/m3s', 'mol/cm3s', 'wppm/s')
        
        Returns:
            Delta C in target units
        """
        delta_c = np.array(delta_c)
        
        if to_unit == 'mol/m\u00b3s':
            return delta_c
        elif to_unit == 'mol/cm\u00b3s':
            return delta_c * 1e-6 
        elif to_unit == 'wppm/s':
            coef = (1.01/self.atw) * (self.atw/self.mass_density)
            return delta_c * coef
        else:
            raise ValueError(f"Unknown delta C unit: {to_unit}")
    
    def delta_c_to_flux(self, delta_c, thickness=None):
        """
        Convert delta_c (mol/m3s) to flux (mol/m2s).
        
        Args:
            delta_c: Delta C array/value in mol/m3s
            thickness: Material thickness in m (uses self.thickness if None)
        
        Returns:
            Flux in mol/m2s
        """
        delta_c = np.array(delta_c)
        thickness_val = thickness if thickness is not None else self.thickness
        
        flux = delta_c * thickness_val / 2
        
        return flux
    
    def flux_to_delta_c(self, flux, thickness=None):
        """
        Convert flux (mol/m2s) to delta_c (mol/m3s).
        
        Args:
            flux: Flux array/value in mol/m2s
            thickness: Material thickness in m (uses self.thickness if None)
        
        Returns:
            Delta C in mol/m3s
        """
        flux = np.array(flux)
        thickness_val = thickness if thickness is not None else self.thickness
        
        delta_c = flux * 2 / thickness_val
        
        return delta_c
    
    def convert_experimental_to_standard(self, temp, y_data, temp_unit, y_unit, y_type):
        """
        Convert experimental data to standard simulation units.
        
        Args:
            temp: Temperature data
            y_data: Y-axis data (flux or delta_c)
            temp_unit: Temperature unit ('K' or 'degree C')
            y_unit: Y-axis unit (flux or delta_c units)
            y_type: Type of y-data ('flux' or 'delta_c')
        
        Returns:
            tuple: (temperature_K, flux_mol_per_m2_per_s)
        """
        temp_std = self.temperature_to_standard(temp, temp_unit)
        
        if y_type.lower() == 'flux':
            # Convert flux to standard units
            flux_std = self.flux_to_standard(y_data, y_unit)
        elif y_type.lower() == 'delta_c':
            # Convert delta_c to standard units, then to flux
            delta_c_std = self.delta_c_to_standard(y_data, y_unit)
            flux_std = self.delta_c_to_flux(delta_c_std)
        else:
            raise ValueError(f"Unknown y_type: {y_type}. Must be 'flux' or 'delta_c'")
        
        return temp_std, flux_std
    
    def convert_for_display(self, temp, flux, temp_unit, flux_unit):
        """
        Convert standard simulation data for display.
        
        Args:
            temp: Temperature data in K
            flux: Flux data in mol/m2/s
            temp_unit: Desired temperature unit ('K' or 'degree C')
            flux_unit: Desired flux unit ('mol/m2s', 'mol/cm2s', 'wppm m/s')
        
        Returns:
            tuple: (converted_temp, converted_flux)
        """
        temp_display = self.temperature_from_standard(temp, temp_unit)
        flux_display = self.flux_from_standard(flux, flux_unit)
        
        return temp_display, flux_display
    
    def convert_flux_for_display_as_delta_c(self, temp, flux, temp_unit, delta_c_unit):
        """
        Convert standard simulation data (flux) for display as delta_c.
        
        Args:
            temp: Temperature data in K
            flux: Flux data in mol/m2/s
            temp_unit: Desired temperature unit ('K' or 'degree C')
            delta_c_unit: Desired delta_c unit ('mol/m3s', 'mol/cm3s', 'wppm/s')
        
        Returns:
            tuple: (converted_temp, converted_delta_c)
        """
        temp_display = self.temperature_from_standard(temp, temp_unit)
        
        delta_c_std = self.flux_to_delta_c(flux)
        delta_c_display = self.delta_c_from_standard(delta_c_std, delta_c_unit)
        
        return temp_display, delta_c_display
    
    def get_labels(self, temp_unit, y_unit=None, y_type=None, flux_unit=None, delta_c_unit=None):
        """
        Get formatted labels for plotting.
        
        Args:
            temp_unit: Temperature unit
            y_unit: Y-axis unit (when y_type is specified)
            y_type: Type of y-data ('flux' or 'delta_c')
            flux_unit: Flux unit (legacy parameter)
            delta_c_unit: Delta C unit (legacy parameter)
        
        Returns:
            dict: Dictionary with formatted labels
        """
        labels = {}
        
        if temp_unit == 'K':
            labels['temperature'] = 'Temperature [K]'
        else:
            labels['temperature'] = 'Temperature [\u00b0C]'
        
        if y_type and y_unit:
            if y_type.lower() == 'flux':
                labels['y_axis'] = self._get_flux_label(y_unit)
            elif y_type.lower() == 'delta_c':
                labels['y_axis'] = self._get_delta_c_label(y_unit)
        
        if flux_unit:
            labels['flux'] = self._get_flux_label(flux_unit)
        
        if delta_c_unit:
            labels['delta_c'] = self._get_delta_c_label(delta_c_unit)
        
        return labels
    
    def _get_flux_label(self, flux_unit):
        """Get formatted flux label"""
        if flux_unit == 'mol/m\u00b2s':
            return 'Flux [mol/m\u00b2s]'
        elif flux_unit == 'mol/cm\u00b2s':
            return 'Flux [mol/cm\u00b2s]'
        elif flux_unit == 'wppm m/s':
            return 'Flux [wppm m/s]'
        else:
            return f'Flux [{flux_unit}]'
    
    def _get_delta_c_label(self, delta_c_unit):
        """Get formatted delta C label"""
        if delta_c_unit == 'mol/m3s':
            return ' Delta C [mol/m\u00b3s]'
        elif delta_c_unit == 'mol/cm3s':
            return 'Delta C [mol/cm\u00b3s]'
        elif delta_c_unit == 'wppm/s':
            return 'Delta C [wppm/s]'
        else:
            return f'Delta C [{delta_c_unit}]'
    
    def time_from_temperature(self, temp, min_temp, heating_rate):
        """
        Convert temperature to time using heating rate.
        
        Args:
            temp: Temperature array in K
            min_temp: Minimum temperature in K
            heating_rate: Heating rate in K/s
        
        Returns:
            Time array in seconds
        """
        return (temp - min_temp) / heating_rate
    
    def temperature_from_time(self, time, min_temp, heating_rate):
        """
        Convert time to temperature using heating rate.
        
        Args:
            time: Time array in seconds
            min_temp: Minimum temperature in K
            heating_rate: Heating rate in K/s
        
        Returns:
            Temperature array in K
        """
        return min_temp + time * heating_rate